export const useThemeColor = () => {
  const defaultColor = "#1c1c1c";

  const getCurrentThemeColor = (): string => {
    if (import.meta.client) {
      return localStorage.getItem("themeColor") || defaultColor;
    }
    return defaultColor;
  };

  const setThemeColor = (color: string) => {
    if (import.meta.client) {
      const user = useSupabaseUser();
      if (user.value?.user_metadata.role === "organizer") {
        localStorage.setItem("themeColor", color);
        updateCSSVariables(color);
      }
    }
  };

  const updateCSSVariables = (color: string) => {
    if (import.meta.client) {
      const root = document.documentElement;
      root.style.setProperty("--theme-primary", color);
      root.style.setProperty("--theme-primary-rgb", hexToRgb(color));
      root.style.setProperty("--theme-gradient-from", color);
      root.style.setProperty("--theme-gradient-to", lightenColor(color, 10));
      root.style.setProperty("--theme-color", color);

      // Generate color variations
      const variations = generateColorVariations(color);
      Object.entries(variations).forEach(([key, value]) => {
        root.style.setProperty(`--theme-${key}`, value);
      });
    }
  };

  // Lighten color for gradient effect
  const lightenColor = (hex: string, percent: number) => {
    const num = parseInt(hex.replace("#", ""), 16);
    const amt = Math.round(2.55 * percent);
    const R = (num >> 16) + amt;
    const G = ((num >> 8) & 0x00ff) + amt;
    const B = (num & 0x0000ff) + amt;
    return (
      "#" +
      (
        0x1000000 +
        (R < 255 ? (R < 1 ? 0 : R) : 255) * 0x10000 +
        (G < 255 ? (G < 1 ? 0 : G) : 255) * 0x100 +
        (B < 255 ? (B < 1 ? 0 : B) : 255)
      )
        .toString(16)
        .slice(1)
    );
  };

  const hexToRgb = (hex: string): string => {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    if (result) {
      return `${parseInt(result[1], 16)}, ${parseInt(result[2], 16)}, ${parseInt(result[3], 16)}`;
    }
    return "151, 145, 184";
  };

  const generateColorVariations = (baseColor: string) => {
    // Convert hex to HSL for easier manipulation
    const hsl = hexToHsl(baseColor);
    if (!hsl) return {};

    return {
      "50": hslToHex(hsl.h, Math.max(hsl.s - 5, 0), Math.min(hsl.l + 45, 95)),
      "100": hslToHex(hsl.h, Math.max(hsl.s - 5, 0), Math.min(hsl.l + 35, 90)),
      "200": hslToHex(hsl.h, hsl.s, Math.min(hsl.l + 25, 85)),
      "300": hslToHex(hsl.h, hsl.s, Math.min(hsl.l + 15, 75)),
      "400": hslToHex(hsl.h, hsl.s, Math.min(hsl.l + 5, 65)),
      "500": baseColor,
      "600": hslToHex(hsl.h, hsl.s, Math.max(hsl.l - 5, 35)),
      "700": hslToHex(hsl.h, hsl.s, Math.max(hsl.l - 15, 25)),
      "800": hslToHex(hsl.h, hsl.s, Math.max(hsl.l - 25, 15)),
      "900": hslToHex(hsl.h, hsl.s, Math.max(hsl.l - 35, 5)),
    };
  };

  const hexToHsl = (hex: string) => {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    if (!result) return null;

    let r = parseInt(result[1], 16) / 255;
    let g = parseInt(result[2], 16) / 255;
    let b = parseInt(result[3], 16) / 255;

    const max = Math.max(r, g, b);
    const min = Math.min(r, g, b);
    let h = 0,
      s = 0,
      l = (max + min) / 2;

    if (max !== min) {
      const d = max - min;
      s = l > 0.5 ? d / (2 - max - min) : d / (max + min);

      switch (max) {
        case r:
          h = (g - b) / d + (g < b ? 6 : 0);
          break;
        case g:
          h = (b - r) / d + 2;
          break;
        case b:
          h = (r - g) / d + 4;
          break;
      }
      h /= 6;
    }

    return { h: h * 360, s: s * 100, l: l * 100 };
  };

  const hslToHex = (h: number, s: number, l: number): string => {
    h = h / 360;
    s = s / 100;
    l = l / 100;

    const hue2rgb = (p: number, q: number, t: number) => {
      if (t < 0) t += 1;
      if (t > 1) t -= 1;
      if (t < 1 / 6) return p + (q - p) * 6 * t;
      if (t < 1 / 2) return q;
      if (t < 2 / 3) return p + (q - p) * (2 / 3 - t) * 6;
      return p;
    };

    let r, g, b;

    if (s === 0) {
      r = g = b = l;
    } else {
      const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
      const p = 2 * l - q;
      r = hue2rgb(p, q, h + 1 / 3);
      g = hue2rgb(p, q, h);
      b = hue2rgb(p, q, h - 1 / 3);
    }

    const toHex = (c: number) => {
      const hex = Math.round(c * 255).toString(16);
      return hex.length === 1 ? "0" + hex : hex;
    };

    return `#${toHex(r)}${toHex(g)}${toHex(b)}`;
  };

  const initializeTheme = () => {
    if (import.meta.client) {
      const user = useSupabaseUser();
      if (user.value?.user_metadata.role === "organizer") {
        const savedColor = localStorage.getItem("themeColor");
        if (savedColor) {
          updateCSSVariables(savedColor);
        } else {
          // Set default organizer theme if no saved color
          updateCSSVariables(defaultColor);
        }
      }
    }
  };

  return {
    getCurrentThemeColor,
    setThemeColor,
    updateCSSVariables,
    initializeTheme,
    lightenColor,
    hexToRgb,
  };
};
