export const formatDate = (date: Date | string | null | undefined) => {
  if (!date) return '';
  return new Date(date).toLocaleString('default', {
    day: 'numeric',
    month: 'short',
    year: 'numeric',
  });
};

export const formatDateTime = (date: Date | string | null | undefined) => {
  if (!date) return '';
  return new Date(date).toLocaleString('default', {
    day: 'numeric',
    month: 'short',
    year: 'numeric',
    hour: 'numeric',
    minute: 'numeric',
  });
};

export const imgSrc = (src: string | null | undefined, bustCache = false) => {
  if (!src || typeof src !== 'string' || src.trim().length === 0) {
    return '/img/placeholder.png';
  }

  const trimmed = src.trim();

  // If already an absolute URL, return as-is (optionally bust cache)
  if (/^https?:\/\//i.test(trimmed)) {
    if (bustCache) {
      const separator = trimmed.includes('?') ? '&' : '?';
      return `${trimmed}${separator}v=${Date.now()}`;
    }
    return trimmed as string;
  }

  // Build URL from key using public runtime config when available
  const config = useRuntimeConfig();
  const publicCfg: any = (config as any).public || {};
  const mediaBase: string | undefined = publicCfg.MEDIA_BASE_URL;
  const s3PublicBase: string | undefined = publicCfg.S3_PUBLIC_BUCKET_URL;

  const base = mediaBase || s3PublicBase || 'https://palmy-io.s3.us-east-1.amazonaws.com';
  const url = `${base.replace(/\/$/, '')}/${trimmed.replace(/^\//, '')}`;

  if (bustCache) {
    return `${url}?v=${Date.now()}`;
  }
  return url as string;
};

export const setTheme = (primary: string, secondary: string, darkPrimary?: string, darkSecondary?: string) => {
  const el = document.documentElement;
  // Only set the 4 source variables — CSS handles the light/dark swap
  el.style.setProperty('--ui-light-primary', primary);
  el.style.setProperty('--ui-light-secondary', secondary);
  el.style.setProperty('--ui-dark-primary', darkPrimary || primary);
  el.style.setProperty('--ui-dark-secondary', darkSecondary || secondary);
  // Remove any inline --ui-primary/--ui-secondary so .dark CSS rule wins
  el.style.removeProperty('--ui-primary');
  el.style.removeProperty('--ui-secondary');
};

export const resetTheme = () => {
  const defaults = {primary: '#ffcc15', secondary: '#000000', darkPrimary: '#ffcc15', darkSecondary: '#ffcc15'};
  setTheme(defaults.primary, defaults.secondary, defaults.darkPrimary, defaults.darkSecondary);
  return defaults;
};
