import {serverSupabaseSession, serverSupabaseUser} from '#supabase/server';
import {deleteCookie, sendRedirect} from 'h3';
import {prisma} from '~~/server/utils/prisma';

const PROTECTED_SECTIONS = ['/api/admin', '/api/organizer', '/api/user'];

export default defineEventHandler(async event => {
  const url = event.node.req?.url || '';
  const isProtected = PROTECTED_SECTIONS.some(section => url.startsWith(section));

  try {
    const session = await serverSupabaseSession(event);
    const user = await serverSupabaseUser(event);
    const ROLE = user?.user_metadata?.role || 'user';

    // If no session & accessing protected route → redirect
    if ((!session?.access_token || !user?.id) && isProtected) {
      deleteCookies(event);
      return await sendRedirect(event, '/auth/signin', 302);
    }

    // If user exists
    if (user?.id) {
      // Admin bypass
      if (ROLE === 'admin') return;

      const userData = await prisma.user.findUnique({
        where: {uid: user.id},
        select: {
          id: true,
          type: true,
          photo: true,
          email: true,
          permissions: true,
          approved: true,
          theme: true,
        },
      });

      // If user not found in DB & accessing protected route → redirect
      if (!userData && isProtected) {
        deleteCookies(event);
        return await sendRedirect(event, '/auth/signin', 302);
      }

      // Attach user info to context
      if (userData) {
        event.context.userId = userData.id;
        event.context.userRole = ROLE;
        event.context.user = {
          id: userData.id,
          type: userData.type,
          photo: userData.photo,
          email: user.email,
          permissions: userData.permissions,
          approved: userData.approved,
          theme: userData.theme,
        };
      }
    }
  } catch (error) {
    // On error, redirect if protected
    if (isProtected) {
      deleteCookies(event);
      return await sendRedirect(event, '/auth/signin', 302);
    }
  }
});

function deleteCookies(event: any) {
  deleteCookie(event, 'sb-access-token');
  deleteCookie(event, 'sb-refresh-token');
  deleteCookie(event, 'sb-eqfljqigltxlnfnallow-auth-token');
}
