import { defineEventHandler, getQuery, sendRedirect, setCookie } from 'h3';
import jwt from 'jsonwebtoken';

export default defineEventHandler(async (event) => {
  const method = event.method;

  if (method === 'GET') {
    return await authenticateWalletDeepLink(event);
  }
});

async function authenticateWalletDeepLink(event: any) {
  try {
    const query = getQuery(event);
    const token = query.token as string;

    if (!token) {
      throw createError({
        statusCode: 400,
        statusMessage: 'Missing authentication token',
      });
    }

    // Verify the JWT token
    const decoded = jwt.verify(token, process.env.JWT_SECRET || 'fallback-secret') as any;

    if (!decoded.userId || !decoded.couponId || decoded.action !== 'view_profile') {
      throw createError({
        statusCode: 400,
        statusMessage: 'Invalid token payload',
      });
    }

    // Verify the user and coupon still exist and are accessible
    const coupon = await prisma.coupon.findFirst({
      where: {
        id: decoded.couponId,
        projects: {
          some: {
            OR: [{ drawingEnter: { some: { userId: decoded.userId } } }, { contestEnter: { some: { userId: decoded.userId } } }, { gameEnter: { some: { userId: decoded.userId } } }],
          },
        },
      },
      include: {
        user: {
          select: {
            id: true,
            orgName: true,
            photo: true,
          },
        },
      },
    });

    if (!coupon) {
      throw createError({
        statusCode: 404,
        statusMessage: 'Coupon not found or not accessible',
      });
    }

    // Generate a session token for automatic login
    const sessionToken = jwt.sign(
      {
        userId: decoded.userId,
        couponId: decoded.couponId,
        organizerId: coupon.user.id,
        loginType: 'wallet_deeplink',
        exp: Math.floor(Date.now() / 1000) + 60 * 60, // 1 hour
      },
      process.env.JWT_SECRET || 'fallback-secret'
    );

    // Set the session cookie and redirect
    setCookie(event, 'wallet-auth-token', sessionToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 60 * 60, // 1 hour
    });

    // Redirect to the organizer's profile page
    return sendRedirect(event, `/user/organization/${coupon.user.id}?from=wallet`);
  } catch (error) {
    console.error('Wallet deep link authentication error:', error);

    if (error.name === 'JsonWebTokenError' || error.name === 'TokenExpiredError') {
      throw createError({
        statusCode: 401,
        statusMessage: 'Invalid or expired authentication token',
      });
    }

    throw createError({
      statusCode: 500,
      statusMessage: 'Authentication failed',
    });
  }
}
