import OpenAI from 'openai';

const config = useRuntimeConfig();

const GOOGLE_API_KEY = config.GOOGLE_API_KEY;
const GOOGLE_CX_ID = config.GOOGLE_CX_ID;
const OPENAI_API_KEY = config.OPENAI_API_KEY;

const openai = new OpenAI({
  apiKey: OPENAI_API_KEY,
});

export default defineEventHandler(async event => {
  const body = await readBody(event);
  if (!body?.topic) {
    return {status: false, message: 'Missing topic'};
  }
  const getPrompts = await prisma.settings.findUnique({
    where: {key: 'prompts'},
    select: {value: true},
  });
  if (!getPrompts) return {status: false, message: 'Missing prompts'};

  const prompts = JSON.parse(getPrompts.value);

  let prompt = body.type === 'Trivia' ? prompts.trivia : prompts.poll;
  prompt = prompt.replace('[[user_topic]]', body.topic);

  try {
    const result = await generateMCQ(prompt);
    if (!result) return {status: false, message: `Failed to generate ${body.type}. Please try again later.`};
    if (body.type !== 'Trivia') {
      await Promise.all(
        result.answers.map(async (answer: any, key: number) => {
          const imageUrl = await fetchImageForAnswer(answer.answer, body.topic);
          if (imageUrl) result.answers[key].photo = imageUrl;
        })
      );
    }

    return {status: true, question: result.question, answers: result.answers};
  } catch (e: any) {
    return {status: false, message: `Failed to generate ${body.type}. Please try again later.`};
  }
});

async function generateMCQ(prompt: string) {
  const response = await openai.chat.completions.create({
    messages: [{role: 'system', content: prompt}],
    model: 'gpt-4.1-mini',
    temperature: 0.7,
    max_tokens: 250,
    response_format: {type: 'json_object'},
  });

  try {
    const jsonText = JSON.parse(response.choices[0].message.content);
    if (typeof jsonText !== 'object') return false;
    return jsonText;
  } catch (err) {
    return false;
  }
}

const fetchImageForAnswer = async (query: string, topic: string) => {
  try {
    let imgUrl = await fatchImage(topic + ': ' + query);
    const res = await fetch(imgUrl);
    if (!res.ok) {
      imgUrl = await fatchImage(query);
    }

    return imgUrl;
  } catch (e) {
    return '';
  }
};

const fatchImage = async (query: string) => {
  try {
    const res = await fetch(
      `https://www.googleapis.com/customsearch/v1?q=${encodeURIComponent(query)}&searchType=image&imgSize=small&key=${GOOGLE_API_KEY}&cx=${GOOGLE_CX_ID}`
    );
    const data = await res.json();
    return data.items?.[0]?.link || '';
  } catch (e) {
    return '';
  }
};
