import OpenAI from 'openai';

const config = useRuntimeConfig();

const GOOGLE_API_KEY = config.GOOGLE_API_KEY;
const GOOGLE_CX_ID = config.GOOGLE_CX_ID;
const OPENAI_API_KEY = config.OPENAI_API_KEY;

const openai = new OpenAI({
  apiKey: OPENAI_API_KEY,
});

export default defineEventHandler(async (event) => {
  const body = await readBody(event);
  if (!body?.topic || !body?.type) {
    return { status: false, message: 'Missing topic or type' };
  }
  const promptData = await prisma.settings.findUnique({
    where: { key: body.type === 'trivia' ? 'triviaPrompt' : body.type === 'survey' ? 'pollPrompt' : 'songPrompt' },
    select: { value: true },
  });
  if (!promptData) return { status: false, message: 'Missing prompts' };
  const prompt = promptData.value;

  try {
    const result = await generateMCQ(prompt, body.type, body.topic);
    if (!result) return { status: false, message: `Failed to generate ${body.type}. Please try again later.` };
    if (!result.question || !Array.isArray(result.answers)) {
      return { status: false, message: `Failed to generate ${body.type}. Please try again later.` };
    }
    if (body.type === 'survey' || body.type === 'song') {
      await Promise.all(
        result.answers.map(async (answer: any, key: number) => {
          const imageUrl = await fetchImageForAnswer(answer.answer, body.topic);
          if (imageUrl) result.answers[key].photoUrl = imageUrl;
        })
      );
    }

    return { status: true, question: result.question, answers: result.answers };
  } catch (e: any) {
    return { status: false, message: `Failed to generate ${body.type}. Please try again later.`, error: e };
  }
});

async function generateMCQ(prompt: string, type: string, topic: string) {
  try {
    const tools = getTools(type);
    const response = await openai.chat.completions.create({
      model: 'gpt-4-1106-preview', // ✅ required model
      temperature: 0.7,
      messages: [
        { role: 'system', content: prompt },
        { role: 'user', content: `Topic: ${topic}` },
      ],
      tools, // ✅ Correct key
      tool_choice: {
        type: 'function',
        function: {
          name: 'generate_contest', // ✅ Must match name in tools[]
        },
      },
    });

    return JSON.parse(response.choices[0].message.tool_calls[0].function.arguments);
  } catch (e) {
    return false;
  }
}

const fetchImageForAnswer = async (query: string, topic: string) => {
  try {
    let imgUrl = await fatchImage(topic + ': ' + query);
    const res = await fetch(imgUrl);
    if (!res.ok) {
      imgUrl = await fatchImage(query);
    }

    return imgUrl;
  } catch (e) {
    return '';
  }
};

const fatchImage = async (query: string) => {
  try {
    const data = await $fetch(`https://www.googleapis.com/customsearch/v1?q=${encodeURIComponent(query)}&searchType=image&imgSize=medium&key=${GOOGLE_API_KEY}&cx=${GOOGLE_CX_ID}`, {
      headers: {
        'cache-control': 'max-age=60', // cache 60s server-side
      },
    });
    return data.items?.[0]?.link || '';
  } catch (e) {
    return '';
  }
};

// Tools
function getTools(type: string) {
  if (type === 'trivia') return triviaTools;
  if (type === 'survey') return surveyTools;
  if (type === 'song') return songTools;
}

const songTools = [
  {
    type: 'function',
    function: {
      name: 'generate_contest',
      description: 'Generates a high-energy, stadium-style fan poll with 4 opinion-based answer options.',
      parameters: {
        type: 'object',
        properties: {
          question: {
            type: 'string',
            description: 'Casual, energetic poll question under 15 words, tied to the user topic',
          },
          answers: {
            type: 'array',
            description: 'Four diverse, relevant, and family-safe poll options',
            items: {
              type: 'object',
              properties: {
                answer: { type: 'string' },
                artist: { type: 'string' },
                photoUrl: {
                  type: 'string',
                  description: 'empty string',
                },
              },
              required: ['answer', 'artist', 'photoUrl'],
            },
            minItems: 4,
            maxItems: 4,
          },
        },
        required: ['question', 'answers'],
      },
    },
  },
];

const triviaTools = [
  {
    type: 'function',
    function: {
      name: 'generate_contest',
      description: 'Generates a high-energy, stadium-style fan poll with 4 opinion-based answer options.',
      parameters: {
        type: 'object',
        properties: {
          question: {
            type: 'string',
            description: 'Casual, energetic poll question under 15 words, tied to the user topic',
          },
          answers: {
            type: 'array',
            description: 'Four diverse, relevant, and family-safe poll options',
            items: {
              type: 'object',
              properties: {
                answer: { type: 'string' },
                correct: { type: 'boolean' },
              },
              required: ['answer', 'correct'],
            },
            minItems: 4,
            maxItems: 4,
          },
        },
        required: ['question', 'answers'],
      },
    },
  },
];

const surveyTools = [
  {
    type: 'function',
    function: {
      name: 'generate_contest',
      description: 'Generates a high-energy, stadium-style fan poll with 4 opinion-based answer options.',
      parameters: {
        type: 'object',
        properties: {
          question: {
            type: 'string',
            description: 'Casual, energetic poll question under 15 words, tied to the user topic',
          },
          answers: {
            type: 'array',
            description: 'Four diverse, relevant, and family-safe poll options',
            items: {
              type: 'object',
              properties: {
                answer: { type: 'string' },
                photoUrl: {
                  type: 'string',
                  description: 'empty string',
                },
              },
              required: ['answer', 'photoUrl'],
            },
            minItems: 4,
            maxItems: 4,
          },
        },
        required: ['question', 'answers'],
      },
    },
  },
];
