type getRequest = {page: number; limit?: number};

export default defineEventHandler(async event => {
  const USER_ID = Number(event.context.user.id);
  const method = event.method;
  if (method == 'GET') {
    const query: getRequest = getQuery(event);
    return await getEvents(query, USER_ID);
  } else if (method == 'POST') {
    const body = await readBody(event);
    const {id, attendance} = body;
    try {
      return await prisma.project
        .update({
          where: {id: Number(id), userId: USER_ID},
          data: {attendance: Number(attendance)},
        })
        .then(() => {
          return {message: 'Attendance updated successfully', status: true};
        })
        .catch((err: any) => {
          return {
            message: 'Error updating attendance',
            status: false,
          };
        });
    } catch (err: any) {
      return {
        message: 'Error updating attendance',
        status: false,
      };
    }
  }
});

async function getEvents(query: getRequest, USER_ID: number) {
  const page = query.page || 1;
  const limit = query.limit || 10;
  const skip = (page - 1) * limit;
  try {
    const events = await prisma.project.findMany({
      where: {
        userId: USER_ID,
        status: {in: ['live', 'completed']},
      },
      select: {
        id: true,
        name: true,
        category: true,
        type: true,
        teamTag: true,
        eventDate: true,
        attendance: true,
        sponsors: {
          select: {
            name: true,
            logo: true,
          },
        },
        fanPhoto: true,
        contestEnter: true,
        drawingEnter: true,
        gameEnter: true,
      },
      skip: skip,
      take: limit,
    });
    const totalCount = await prisma.project.count({
      where: {userId: USER_ID},
    });
    let interactions = 0;
    let attendance = 0;
    const sponsors = {};
    events.forEach(event => {
      event.userEngaged = 0;
      interactions +=
        event.contestEnter.length + event.drawingEnter.length + event.gameEnter.length + event.fanPhoto.length;
      attendance += event.attendance;
      if (event.type === 'fanphoto') {
        event.userEngaged = event.fanPhoto.length || 0;
      } else if (event.category === 'contest') {
        event.userEngaged = event.contestEnter.length || 0;
      } else if (event.category === 'drawing') {
        event.userEngaged = event.drawingEnter.length || 0;
      } else if (event.category === 'game') {
        event.userEngaged = event.gameEnter.length || 0;
      }
      event.sponsors.forEach(sponsor => {
        if (sponsors[sponsor.name]) {
          sponsors[sponsor.name]++;
        } else {
          sponsors[sponsor.name] = 1;
        }
      });
    });

    const topSponsors = Object.keys(sponsors).reduce((a, b) => (sponsors[a] > sponsors[b] ? a : b), '');

    return {
      events,
      attendance,
      interaction: interactions,
      topSponsor: topSponsors,
      totalCount,
      page,
      totalPages: Math.ceil(totalCount / limit),
    };
  } catch (err: any) {
    return {message: 'Error fetching coupons', error: err.message};
  }
}
