import {ApiResponse} from '~~/server/utils/apiResponse';

export default defineEventHandler(async event => {
  const USER = event.context.user;
  const USER_ID = Number(USER.id);

  const USER_PERMISSIONS = USER.permissions || {};
  if (!USER_PERMISSIONS.contest)
    return apiResponse({status: false, message: 'You do not have permission to create contests.'});
  const form = await readFormData(event);
  return await addProject(form, USER_ID);
});

async function addProject(form: FormData, USER_ID: number) {
  const project = JSON.parse(form.get('project') as string);
  const photo = form.get('photo') as unknown as File;

  project.expiry = project.expiry ? new Date(project.expiry) : null;
  project.category = 'contest';
  project.status = 'queue';

  try {
    if (photo) {
      await s3Upload(photo, USER_ID).then(({status, url}) => {
        if (status && url) project.photo = url;
      });
    }

    const result = await prisma.project.create({
      data: {
        category: 'contest',
        photos: [],
        photo: project.photo,
        name: project.name,
        liveTitle: project.liveTitle || '',
        qrTitle: project.qrTitle || '',
        qrSubTitle: project.qrSubTitle || '',
        teamTag: project.teamTag || null,
        eventDate: project.eventDate ? new Date(project.eventDate) : null,
        eventName: project.eventName || null,
        distribution: project.distribution,
        result: project.result || '',
        delayed: project.delayed,
        notify: project.notify,
        sponsors: {
          connect: project.sponsors.map((sponsor: string) => ({
            id: Number(sponsor),
          })),
        },
        coupons: {
          connect: project.coupons.map((coupon: string) => ({
            id: Number(coupon),
          })),
        },
        status: 'queue',
        user: {connect: {id: USER_ID}},
        type: project.type || '',
        approved: true,
      },
    });

    if (result) {
      if (project.contests.length > 0) {
        Object.entries(project.contests).forEach(async ([key, contest]) => {
          let image = '';
          const contestImage = form.get('quesPhoto') as unknown as File;
          if (contestImage) {
            await s3Upload(contestImage, USER_ID).then(ret => {
              if (ret.status && ret.url) image = ret.url;
            });
          }
          if (contest.answers) {
            await Promise.all(
              contest.answers.map(async (answer, anskey) => {
                const answersImage = form.get('answerImage' + key + anskey) as unknown as File;
                if (answersImage) {
                  await s3Upload(answersImage, USER_ID).then(ret => {
                    if (ret.status && ret.url) {
                      answer.photo = ret.url;
                      answer.photoUrl = '';
                    }
                  });
                }
              })
            );
          }
          await prisma.contest.create({
            data: {
              question: contest.question,
              image: image,
              project: {connect: {id: result.id}},
              answers: {
                create: contest.answers.map(answer => ({
                  photoUrl: answer.photoUrl || '',
                  photo: answer.photo || '',
                  answer: answer.answer,
                  correct: answer.correct,
                  artist: answer.artist,
                })),
              },
            },
          });
        });
      }
      return apiResponse({result: result.id});
    }
    throw new Error('Failed to create project');
  } catch (error) {
    return apiResponse({
      status: false,
      message: 'Error uploading photo',
      error,
    });
  }
}
