export default defineEventHandler(async event => {
  const USER = event.context.user;
  if (!USER.permissions.drawing)
    return apiResponse({status: false, message: 'You do not have permission to create drawings.'});

  const form = await readFormData(event);
  return await addProject(form, USER.id);
});

async function addProject(form: FormData, USER_ID: number) {
  const project = JSON.parse(form.get('project') as string);
  const photo = form.get('photo') as File;
  project.expiry = project.expiry ? new Date(project.expiry) : null;
  if (photo) {
    await s3Upload(photo, USER_ID).then(({status, url}) => {
      if (status && url) project.photo = url;
    });
  }
  try {
    const result = await prisma.project.create({
      data: {
        category: 'drawing',
        photos: [],
        photo: project.photo,
        name: project.name,
        liveTitle: project.liveTitle,
        qrSubTitle: project.qrSubTitle,
        teamTag: project.teamTag || null,
        eventDate: project.eventDate ? new Date(project.eventDate) : null,
        eventName: project.eventName || null,
        notify: project.notify,
        sponsors: {
          connect: project.sponsors.map((s: {id: number}) => ({
            id: Number(s.id),
          })),
        },
        coupons: {
          connect: project.coupons.map((c: {id: number}) => ({
            id: Number(c.id),
          })),
        },
        drawing: {
          create: {
            prize: project.drawing.prize,
            prizeType: project.drawing.prizeType,
          },
        },
        distribution: project.distribution,
        delayed: project.delayed,
        status: 'queue',
        user: {connect: {id: USER_ID}},
        type: project.type || 'drawing',
        description: project.description || '',
        approved: true,
      },
    });
    if (result) return apiResponse({result: result.id});
    throw new Error('');
  } catch (error) {
    if (project.photo) await s3Delete(project.photo).catch(() => null);

    return apiResponse({
      status: false,
      message: 'Something went wrong. Please try again later.',
      error,
    });
  }
}
