import {defineEventHandler, getQuery} from 'h3';
type getRequest = {
  page: number;
  limit?: number;
};

export default defineEventHandler(async event => {
  const USER_ID = Number(event.context.user.id);

  const method = event.method;
  if (method == 'GET') {
    const query: getRequest = getQuery(event);
    return await getProjects(query, USER_ID);
  } else if (method == 'POST') {
    const USER_PERMISSIONS = event.context.userPermissions || {};
    if (!USER_PERMISSIONS.game) return {status: false, message: 'You do not have permission to create game.'};

    const form = await readFormData(event);
    return await createProject(USER_ID, form);
  }
});

async function getProjects(query: getRequest, USER_ID: number) {
  const page = query.page || 1;
  const limit = query.limit || 10;
  const skip = (page - 1) * limit;
  try {
    const projects = await prisma.project.findMany({
      where: {
        userId: USER_ID,
      },
      select: {
        id: true,
        photos: true,
        name: true,
        category: true,
        type: true,
        status: true,
        game: true,
        sponsors: true,
        coupons: true,
        description: true,
        distribution: true,
        delayed: true,
        notify: true,
        target: true,
      },
      skip: skip,
      take: limit,
    });
    const totalCount = await prisma.project.count({
      where: {userId: USER_ID},
    });
    return {
      projects,
      totalCount,
      page,
      totalPages: Math.ceil(totalCount / limit),
    };
  } catch (err: any) {
    return {
      message: 'Error fetching projects',
      error: err.message,
    };
  }
}

async function createProject(USER_ID: number, form: any) {
  const project = JSON.parse(form.get('project') as string);
  const photo = form.get('photo') as File | null;
  const bingoMarker = form.get('bingoMarker') as File | null;
  const gamePhotos = form.getAll('gamePhotos') as File[] | null;

  project.expiry = project.expiry ? new Date(project.expiry) : null;
  project.category = 'game';

  try {
    project.photos = [];

    if (project.type === 'bingo' && project.bingoActions && Array.isArray(project.bingoActions)) {
      project.bingoActions = project.bingoActions.filter(
        (action: any) => action && action.action && action.action.trim()
      );
    }

    if (photo) {
      await s3Upload(photo, USER_ID).then((ret: any) => {
        if (ret.status && ret.url) project.photos.push(ret.url);
      });
    }

    if (project.type === 'bingo' && bingoMarker) {
      await s3Upload(bingoMarker, USER_ID).then((ret: any) => {
        if (ret.status && ret.url) project.game.photo = ret.url;
      });
    }

    // Generate default eventName if not provided
    let eventName = project.eventName;
    if (!eventName && project.teamTag && project.eventDate) {
      const date = new Date(project.eventDate);
      const formattedDate = date.toLocaleDateString('en-US', {
        month: '2-digit',
        day: '2-digit',
        year: 'numeric',
      });
      eventName = `${project.teamTag} - ${formattedDate}`;
    }

    return await prisma.project
      .create({
        data: {
          name: project.name,
          photos: project.photos,
          category: project.category,
          distribution: project.distribution,
          target: project.target ?? null,
          liveTitle: project.liveTitle || '',
          qrTitle: project.qrTitle || '',
          description: project.description ?? '',
          teamTag: project.teamTag || null,
          eventDate: project.eventDate ? new Date(project.eventDate) : null,
          eventName: eventName || null,
          delayed: project.delayed,
          notify: project.notify,
          coupons: {
            connect: (project.coupons || []).map((coupon: string) => ({
              id: Number(coupon),
            })),
          },
          sponsors: {
            connect: (project.sponsors || []).map((sponsor: string) => ({
              id: Number(sponsor),
            })),
          },
          status: 'queue',
          userId: USER_ID,
          type: project.type || '',
          expiry: project.expiry,
          approved: true,
        },
      })
      .then(async data => {
        if (project.type === 'target') {
          await prisma.game.create({
            data: {
              answer: 0,
              projectId: data.id,
              userId: USER_ID,
            },
          });
        }
        if (project.type === 'bingo') {
          await prisma.game.create({
            data: {
              photo: project.game.photo,
              bingoActions: project.game.bingoActions || null,
              bingoType: project.game.bingoType || '',
              answer: 0,
              projectId: data.id,
              userId: USER_ID,
            },
          });
        }
        if (project.type === 'rateit' && project.games) {
          let key = 0;
          project.games.forEach(async (game: any) => {
            if (gamePhotos && gamePhotos[key]) {
              await s3Upload(gamePhotos[key], USER_ID).then((ret: any) => {
                if (ret.status && ret.url) game.photo = ret.url;
              });
            }
            await prisma.game.create({
              data: {
                photo: game.photo,
                text: game.text,
                question: game.question,
                answer: game.answer,
                video: game.video,
                projectId: data.id,
                userId: USER_ID,
              },
            });
            key++;
          });
        }
        if (project.type === 'lottery') {
          await prisma.game.create({
            data: {
              projectId: data.id,
              userId: USER_ID,
            },
          });
        }
        return data ? {status: true} : {status: false};
      })
      .catch(async err => {
        if (project.photos && project.photos.length > 0) {
          project.photos.forEach(async (image: string) => {
            await s3Delete(image);
          });
        }
        if (project.game?.photo && project.game.photo.length > 0) {
          project.game.photo.forEach(async (image: string) => {
            await s3Delete(image);
          });
        }
        return {
          status: false,
          message: 'Error adding Project',
          error: err.message,
        };
      });
  } catch (err: any) {
    return {
      status: false,
      message: err.message || 'Error creating project',
    };
  }
}
