import {defineEventHandler} from 'h3';
import {s3Delete, s3Upload} from '~~/server/utils/s3Actions.post';

import {supabaseAdmin} from '~~/server/utils/supabase';

export default defineEventHandler(async event => {
  const USER_ID = Number(event.context.user.id);

  const Profile = await prisma.user.findUnique({
    where: {id: USER_ID},
    select: {
      id: true,
      uid: true,
      photo: true,
      orgName: true,
      file: true,
      bio: true,
      email: true,
      phone: true,
      isd: true,
      address: true,
      city: true,
      state: true,
      zip: true,
      contact: {
        select: {
          name: true,
          email: true,
          dob: true,
          govId: true,
          ssn: true,
        },
      },
      bank: {
        select: {
          name: true,
          account: true,
          routing: true,
        },
      },
    },
  });
  if (!Profile) return apiResponse({status: false, message: 'Unauthorized request'});

  const method = event.method;
  if (method == 'GET') {
    return apiResponse({result: Profile});
  }
  if (method == 'PUT') {
    const supabase = supabaseAdmin();
    const form = await readFormData(event);
    const user = JSON.parse(form.get('profile') as string);
    const file = form.get('file') as File | null;
    const photo = form.get('photo') as File | null;
    const govId = form.get('govId') as File | null;
    const oldFile = Profile.file ?? '';
    const oldPhoto = Profile.photo ?? '';
    const oldGovId = Profile.contact?.govId ?? '';
    const UUID = Profile.uid;

    // Ensure user.contact exists for govId upload
    if (!user.contact) user.contact = {};

    try {
      if (file) {
        await s3Upload(file, USER_ID).then((ret: any) => {
          if (ret.status && ret.url) user.file = ret.url;
          if (oldFile && oldFile != '') s3Delete(oldFile);
        });
      }
      if (photo) {
        await s3Upload(photo, USER_ID).then((ret: any) => {
          if (ret.status && ret.url) user.photo = ret.url;
          if (oldPhoto && oldPhoto != '') s3Delete(oldPhoto);
        });
      }

      if (govId) {
        await s3Upload(govId, USER_ID).then((ret: any) => {
          if (ret.status && ret.url) user.contact.govId = ret.url;
          if (oldGovId && oldGovId != '') s3Delete(oldGovId);
        });
      }

      const contactData = {
        name: user.contact.name,
        email: user.contact.email,
        dob: user.contact.dob ? new Date(user.contact.dob) : null,
        govId: user.contact.govId,
        ssn: user.contact.ssn,
      };

      const bankData = {
        name: user.bank?.name,
        account: user.bank?.account,
        routing: user.bank?.routing,
      };

      return await prisma.user
        .update({
          where: {id: USER_ID},
          data: {
            bio: user.bio ?? '',
            photo: user.photo,
            orgName: user.orgName,
            file: user.file,
            email: user.email,
            phone: user.phone,
            isd: user.isd,
            contact: {
              upsert: {
                create: contactData,
                update: contactData,
              },
            },
            bank: {
              upsert: {
                create: bankData,
                update: bankData,
              },
            },
            address: user.address,
            city: user.city,
            state: user.state,
            zip: user.zip,
          },
        })
        .then(async res => {
          await supabase.auth.admin.updateUserById(UUID, {
            user_metadata: {
              photo: user.photo,
            },
          });
          return apiResponse({
            result: {
              photo: user.photo,
              name: user.orgName,
              govId: user.contact.govId,
              file: user.file,
            },
          });
        })
        .catch(err => {
          return apiResponse({
            status: false,
            message: 'Error updating user',
            error: err.message,
          });
        });
    } catch (error: any) {
      return apiResponse({
        status: false,
        message: 'Error verifying user',
        error: error.message,
      });
    }
  }
});
