export default defineEventHandler(async event => {
  const USER_ID = Number(event.context.user.id);
  if (event.method === 'GET') {
    try {
      return await getRoster(USER_ID);
    } catch (e: any) {
      return {status: false, message: 'Database unavailable. Please try again shortly.'};
    }
  }

  if (event.method === 'PUT') {
    const body = await readBody(event).catch(() => ({}) as any);
    try {
      return await upsertRoster(USER_ID, body || {});
    } catch (e: any) {
      return {status: false, message: 'Database unavailable. Please try again shortly.'};
    }
  }

  return {status: false, message: 'Method not allowed'};
});

async function getRoster(userId: number) {
  const [settings, teams] = await Promise.all([
    prisma.rosterSettings.findUnique({where: {userId}, select: {enabled: true, mode: true}}),
    prisma.rosterTeam.findMany({
      where: {userId},
      select: {
        id: true,
        side: true,
        logo: true,
        players: {
          orderBy: {order: 'asc'},
          select: {
            id: true,
            number: true,
            name: true,
            position: true,
            class: true,
            height: true,
            hometown: true,
            bio: true,
            photo: true,
            order: true,
          },
        },
      },
      orderBy: {side: 'asc'},
    }),
  ]);

  return {status: true, settings: settings || {enabled: false, mode: 'home'}, teams};
}

type UpsertInput = {
  settings?: {enabled?: boolean; mode?: 'home' | 'home_away'};
  teams?: Array<{id?: number; side: 'home' | 'away'; logo?: string | null}>;
  players?: Array<
    | {op: 'create'; teamId: number; data: PlayerInput}
    | {op: 'update'; id: number; data: PlayerInput}
    | {op: 'delete'; id: number}
  >;
};

type PlayerInput = {
  number?: number | null;
  name?: string;
  position?: string | null;
  class?: string | null;
  height?: string | null;
  hometown?: string | null;
  bio?: string | null;
  photo?: string | null;
  order?: number;
};

async function upsertRoster(userId: number, body: UpsertInput) {
  const tx: any[] = [];

  if (body.settings) {
    tx.push(
      prisma.rosterSettings.upsert({
        where: {userId},
        update: {enabled: body.settings.enabled ?? undefined, mode: (body.settings.mode as any) ?? undefined},
        create: {userId, enabled: !!body.settings.enabled, mode: (body.settings.mode as any) || 'home'},
      })
    );
  }

  if (body.teams && Array.isArray(body.teams)) {
    for (const t of body.teams) {
      if (t.id) {
        tx.push(prisma.rosterTeam.update({where: {id: t.id}, data: {logo: t.logo ?? undefined}}));
      } else {
        tx.push(
          prisma.rosterTeam.upsert({
            where: {userId_side: {userId, side: t.side as any}},
            update: {logo: t.logo ?? undefined},
            create: {userId, side: t.side as any, logo: t.logo ?? null},
          })
        );
      }
    }
  }

  if (body.players && Array.isArray(body.players)) {
    for (const p of body.players) {
      if (p.op === 'create') {
        tx.push(prisma.rosterPlayer.create({data: {teamId: p.teamId, ...(p.data as any)}}));
      } else if (p.op === 'update') {
        tx.push(prisma.rosterPlayer.update({where: {id: p.id}, data: {...(p.data as any)}}));
      } else if (p.op === 'delete') {
        // Use deleteMany to avoid throwing if the record no longer exists
        tx.push(prisma.rosterPlayer.deleteMany({where: {id: p.id}}));
      }
    }
  }

  if (tx.length === 0) return {status: false, message: 'Nothing to update'};

  await prisma.$transaction(tx);
  return await getRoster(userId);
}
