import { getHeader, readRawBody } from 'h3';
import { stripe } from '~~/server/utils/stripe';

import { supabaseAdmin } from '~~/server/utils/supabase';

export default defineEventHandler(async (event) => {
  const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET!;

  const signature = getHeader(event, 'stripe-signature');
  const rawBody = await readRawBody(event);

  if (!signature || !rawBody) {
    return {
      statusCode: 400,
      statusMessage: 'Missing Stripe signature or body',
    };
  }

  let stripeEvent;

  try {
    stripeEvent = stripe.webhooks.constructEvent(rawBody, signature, endpointSecret);
  } catch (err: any) {
    return {
      statusCode: 400,
      statusMessage: 'Webhook signature verification failed',
    };
  }
  // ✅ Handle events
  switch (stripeEvent.type) {
    case 'charge.succeeded': {
      const charge = stripeEvent.data.object;
      const paymentIntent = charge.payment_intent as string;
      const userEmail = charge.billing_details?.email as string | null;
      if (!paymentIntent || !charge || !userEmail) break;
      const donation = await prisma.donations.findUnique({
        where: { paymentId: paymentIntent },
        select: { id: true, email: true, userId: true },
      });
      if (!donation) break;
      if (!donation.userId) {
        await prisma.user
          .findUnique({
            where: { email: userEmail },
            select: { id: true },
          })
          .then(async (User) => {
            if (!User) {
              await createUser(userEmail).then(async (userId: any) => {
                await updateDonation(donation, charge, userId);
              });
            } else {
              await updateDonation(donation, charge, User.id);
            }
          });
      } else {
        await updateDonation(donation, charge);
      }
      await $fetch('/api/mail/donation/' + paymentIntent);
      break;
    }

    case 'charge.refunded': {
      const charge = stripeEvent.data.object;
      await prisma.donations.update({
        where: { chargeId: charge.id },
        data: { status: 'refunded' },
      });
      break;
    }
  }

  return { received: true };
});

async function updateDonation(donation: any, charge: any, USER_ID: number | null = null) {
  const card = charge.payment_method_details.card;
  const data = {
    chargeId: charge.id,
    status: charge.status === 'succeeded' ? 'paid' : 'failed',
    card: `${card.brand} •••• ${card.last4}`,
    paidAt: new Date(),
    email: charge.billing_details?.email || null,
  };
  if (USER_ID) data['userId'] = USER_ID;
  return await prisma.donations
    .update({
      where: { id: donation.id },
      data: data,
    })
    .then((updatedDonation) => {
      return true;
    })
    .catch((error) => {
      return false;
    });
}

// stripe listen --forward-to localhost:3000/api/user/webhook
async function createUser(uEmail: string) {
  const supabase = supabaseAdmin();
  const supUser = {
    email: uEmail,
    password: 'max123@hunt',
    email_confirm: true,
    user_metadata: {
      role: 'user',
      name: '',
      photo: '',
    },
  };
  return await supabase.auth.admin
    .createUser(supUser)
    .then(async (supdata: any) => {
      const UID = supdata.data.user.id;
      return await prisma.user
        .create({
          data: {
            uid: UID,
            type: 'user',
            email: uEmail,
            themeColor: '',
          },
        })
        .then(async (userInfo) => {
          if (!userInfo) return null;
          return userInfo.id;
        })
        .catch((error) => {
          return null;
        });
    })
    .catch((error) => {
      return null;
    });
}
