// server/api/upload.post.ts
import {DeleteObjectCommand, PutObjectCommand} from '@aws-sdk/client-s3';
import {extname} from 'path';
import sharp from 'sharp';
import {getS3Client} from '~~/server/utils/s3Config';

export const s3Upload = async (
  file: File,
  userId: number
): Promise<{status: boolean; url?: string; error?: string; fullUrl?: string}> => {
  const env = useRuntimeConfig();
  const s3 = getS3Client();

  const ENVIRONMENT = useRuntimeConfig().ENVIRONMENT;
  const TEST_MODE = ENVIRONMENT == 'test' ? true : false;

  if (!file || !userId) {
    return {status: false, error: 'Missing file or userId'};
  }

  const ext = extname(file.name || '');
  const startPath = TEST_MODE ? 'test' : 'user';
  const key = `${startPath}/${userId}/${Date.now()}${ext}`;

  const arrayBuffer = await file.arrayBuffer();
  const buffer = Buffer.from(arrayBuffer);

  // Detect file type
  const isJPEG = file.type === 'image/jpeg' || file.type === 'image/jpg';
  const isPNG = file.type === 'image/png';
  const isWEBP = file.type === 'image/webp';

  // Optimize image using Sharp
  let optimizedBuffer;
  if (isJPEG) {
    optimizedBuffer = await sharp(buffer).jpeg({quality: 80}).toBuffer();
  } else if (isPNG) {
    optimizedBuffer = await sharp(buffer).png({compressionLevel: 8}).toBuffer();
  } else if (isWEBP) {
    optimizedBuffer = await sharp(buffer).webp({quality: 75}).toBuffer();
  } else {
    optimizedBuffer = buffer; // leave non-images untouched
  }

  return await s3
    .send(
      new PutObjectCommand({
        Bucket: env.S3_BUCKET_NAME,
        Key: key,
        Body: optimizedBuffer,
        ContentType: file.type,
        ACL: 'public-read',
      })
    )
    .then(() => {
      const fileUrl = `https://${env.S3_BUCKET_NAME}.s3.${env.AWS_REGION}.amazonaws.com/${key}`;
      return {status: true, url: key, fullUrl: fileUrl};
    })
    .catch(error => {
      return {status: false, error: 'Upload failed', details: error};
    });
};

export const s3Delete = async (key: string) => {
  const env = useRuntimeConfig();
  const s3 = getS3Client();

  if (!key) return {status: false, error: 'Missing key'};

  return await s3
    .send(
      new DeleteObjectCommand({
        Bucket: env.S3_BUCKET_NAME,
        Key: key,
      })
    )
    .then(() => {
      return {status: true, message: 'Deleted successfully'};
    })
    .catch(error => {
      return {status: false, error: 'Delete failed', details: error};
    });
};
